import os
import time
import threading
import requests
from bs4 import BeautifulSoup
from urllib.parse import urljoin, urlparse
from concurrent.futures import ThreadPoolExecutor
from tqdm import tqdm
from pyfiglet import Figlet

# -----------------------------
# GLOBALS
# -----------------------------
downloaded_filenames = set()
filename_lock = threading.Lock()

HEADERS = {
    "User-Agent": "PixieBot/4.1 (+http://pixie.seaverns.com)"
}

# -----------------------------
# UTILS
# -----------------------------
def create_directory(directory):
    os.makedirs(directory, exist_ok=True)

def clear_screen():
    os.system('cls' if os.name == 'nt' else 'clear')

def display_splash():
    figlet = Figlet(font='slant')
    clear_screen()
    print(figlet.renderText('PixieBot 4.1'))
    print("                     Version 4.1.0")
    print("                      By: K0NxT3D")
    time.sleep(3)
    clear_screen()

# -----------------------------
# IMAGE DOWNLOAD
# -----------------------------
def download_image(image_url, folder, progress_bar):
    filename = os.path.basename(image_url.split("?")[0])
    file_path = os.path.join(folder, filename)

    # Thread-safe duplicate check
    with filename_lock:
        if filename in downloaded_filenames or os.path.exists(file_path):
            progress_bar.update(1)
            return
        downloaded_filenames.add(filename)

    try:
        response = requests.get(image_url, headers=HEADERS, timeout=10, stream=True)
        response.raise_for_status()

        with open(file_path, "wb") as f:
            for chunk in response.iter_content(8192):
                if chunk:
                    f.write(chunk)

    except Exception as e:
        print(f"Error downloading {image_url}: {e}")

    progress_bar.update(1)

# -----------------------------
# SCRAPER
# -----------------------------
def get_images_from_page(url, domain, visited):
    images = set()

    try:
        response = requests.get(url, headers=HEADERS, timeout=10)
        response.raise_for_status()

        soup = BeautifulSoup(response.text, "html.parser")

        for img in soup.find_all("img"):
            src = img.get("src")
            if src:
                full_url = urljoin(url, src).split("?")[0]
                if urlparse(full_url).netloc.endswith(domain):
                    images.add(full_url)

    except requests.RequestException:
        pass

    return images

# -----------------------------
# MAIN
# -----------------------------
def main():
    display_splash()

    while True:
        start_url = input("Enter Start URL: ").strip()
        domain = urlparse(start_url).netloc

        print("Scraping images...")
        images = get_images_from_page(start_url, domain, set())

        if not images:
            print("No images found.")
            continue

        save_folder = os.path.join("scraped_images", domain)
        create_directory(save_folder)

        print(f"Found {len(images)} images")

        with tqdm(total=len(images), desc="Downloading") as progress:
            with ThreadPoolExecutor(max_workers=8) as executor:
                for img_url in images:
                    executor.submit(download_image, img_url, save_folder, progress)

        print("Finished!\n")

        if input("Scrape another URL? (y/n): ").lower() != "y":
            break

if __name__ == "__main__":
    main()

