// ============================================================
// HotBox — Network & System Monitor
// ============================================================
//
// Version:        1.0.2
// Author:         K0NxT3D
// Year:           2025
//
// ------------------------------------------------------------
// Overview
// ------------------------------------------------------------
// HotBox is a lightweight, real-time network and system
// monitoring application written in Go. It provides a live,
// browser-based dashboard for observing system health,
// network throughput, and connected WiFi clients.
//
// The application is fully self-contained, embedding its
// HTML, CSS, and JavaScript directly into the Go binary.
// No external frontend build tools or assets are required.
//
// ------------------------------------------------------------
// Primary Use Cases
// ------------------------------------------------------------
// • Monitor CPU, RAM, disk usage, and system temperatures
// • Observe live network RX/TX throughput per interface
// • Detect and display connected WiFi clients with signal strength
// • Provide a clean, local monitoring dashboard for:
//     - Raspberry Pi systems
//     - Home servers
//     - Lab machines
//     - Embedded or headless devices
//
// ------------------------------------------------------------
// Key Features
// ------------------------------------------------------------
// • Real-time metrics via WebSocket streaming (1s updates)
// • Automatic detection of LAN and WiFi interfaces
// • Runtime-selectable network interface
// • WiFi client discovery using `iw station dump`
// • Signal strength tracking per connected client
// • Embedded dark-themed UI (Bootstrap + Orbitron)
// • Zero external runtime dependencies beyond system tools
// • Cross-platform browser auto-launch (Linux, Windows, macOS)
//
// ------------------------------------------------------------
// Architecture & Structure
// ------------------------------------------------------------
// Main Components:
//
// • HTTP Server (Gin)
//     - Serves embedded HTML dashboard
//     - Exposes REST endpoints for device selection
//
// • WebSocket Server (Gorilla WebSocket)
//     - Streams live Metrics struct as JSON
//
// • Metrics Collector
//     - CPU, RAM, Disk via gopsutil
//     - Temperatures via host sensors
//     - Network I/O counters with delta-based rate calculation
//
// • WiFi Monitor Loop
//     - Periodically polls `iw` for connected stations
//     - Extracts MAC addresses and signal strength (dBm)
//
// • Embedded Frontend
//     - Bootstrap-based responsive layout
//     - Progress bars with adaptive coloring
//     - Live device and temperature lists
//
// ------------------------------------------------------------
// Runtime Notes
// ------------------------------------------------------------
// • Requires `iw` to be installed for WiFi client monitoring
// • Must be run with sufficient permissions to access
//   wireless interface data (typically root or CAP_NET_ADMIN)
// • Intended for trusted local or LAN environments
// • WebSocket origin checks are disabled by design
//
// ------------------------------------------------------------
// Network
// ------------------------------------------------------------
// Default HTTP/WebSocket Port: 6969
// Dashboard URL: http://127.0.0.1:6969
//
// ------------------------------------------------------------
// License / Intent
// ------------------------------------------------------------
// Provided for educational, research, and internal monitoring
// purposes. Designed for clarity, extensibility, and minimal
// system overhead.
//
// ============================================================


package main

import (
	"bufio"
	"fmt"
	"log"
	"net/http"
	"os/exec"
	"runtime"
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/gorilla/websocket"

	"github.com/shirou/gopsutil/v3/cpu"
	"github.com/shirou/gopsutil/v3/disk"
	"github.com/shirou/gopsutil/v3/host"
	"github.com/shirou/gopsutil/v3/mem"
	"github.com/shirou/gopsutil/v3/net"
)

const PORT = "6969"

// -----------------------------
// Globals
// -----------------------------
var selectedDevice string
var networkDevices []string
var wifiClients = map[string]int{}
var wifiClientsLock sync.RWMutex

var upgrader = websocket.Upgrader{
	CheckOrigin: func(r *http.Request) bool { return true },
}

var prevNetIO = map[string]net.IOCountersStat{}
var prevNetLock sync.Mutex

// -----------------------------
// Metrics struct
// -----------------------------
type Metrics struct {
	Timestamp   int64              `json:"ts"`
	CPUPercent  float64            `json:"cpu_percent"`
	RAMTotal    uint64             `json:"ram_total"`
	RAMUsed     uint64             `json:"ram_used"`
	RAMPercent  float64            `json:"ram_percent"`
	DiskTotal   uint64             `json:"disk_total"`
	DiskUsed    uint64             `json:"disk_used"`
	DiskPercent float64            `json:"disk_percent"`
	Temps       map[string]float64 `json:"temps"`
	NetRx       uint64             `json:"net_rx"`
	NetTx       uint64             `json:"net_tx"`
	NetRxRate   float64            `json:"net_rx_rate"`
	NetTxRate   float64            `json:"net_tx_rate"`
	WiFiClients map[string]int     `json:"wifi_clients"`
}

// -----------------------------
// HTML Template (embedded)
// -----------------------------
const INDEX_HTML = `
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>HotBox - Network & System Monitor</title>

<!-- Bootstrap CSS -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">

<!-- Orbitron font -->
<link href="https://fonts.googleapis.com/css2?family=Orbitron:wght@400;700&display=swap" rel="stylesheet">

<style>
body {
  background-color: #0c0c0c;
  color: #ff4444;
  font-family: 'Orbitron', monospace;
}
.container { max-width: 1100px; margin: 40px auto; padding: 30px; }
h1, h2, h3 { text-align: center; color: #ff5555; }
h1 { font-size: 2.5rem; }
h2 { font-size: 1.8rem; margin-top: 1.5rem; }
h3 { font-size: 1.2rem; font-weight: 400; margin-bottom: 2rem; }

.card {
  background-color: #1a1a1a;
  border: 1px solid #330000;
  border-radius: 12px;
  box-shadow: 0 0 15px #440000;
  margin-bottom: 20px;
}
.card-header {
  background-color: #220000;
  font-weight: bold;
  font-size: 1.1rem;
  color: #ff6666;
  border-bottom: 1px solid #330000;
}
.card-body { color: #fff; font-size: 0.95rem; }

select.form-select {
  margin-top: 10px;
  background-color: #222;
  color: #fff;
  border-radius: 6px;
  border: 1px solid #330000;
}

ul { list-style: none; padding-left: 0; }
ul li { padding: 6px 10px; margin-bottom: 4px; background-color: #111; border-left: 4px solid #ff2222; border-radius: 5px; }

a { color: #ff6666; text-decoration: none; }
a:hover { color: #ffaaaa; }

#dashboard { display: flex; flex-wrap: wrap; gap: 20px; }
#metrics-card, #clients-card { flex: 1 1 45%; min-width: 300px; }

#darkimg { max-width: 100%; display: block; margin: 0 auto 20px auto; border-radius: 10px; box-shadow: 0 0 20px #330000; }

.progress {
  height: 20px;
  background-color: #111 !important;
  border-radius: 10px;
  margin-bottom: 10px;
  border: 1px solid #330000;
}
.progress-bar {
  background: linear-gradient(90deg, #ff2222, #ff5555);
  transition: width 0.5s ease;
  font-weight: bold;
  color: #fff;
}

@media(max-width: 768px) {
  #metrics-card, #clients-card { flex: 1 1 100%; }
}
</style>
</head>
<body>
<div class="container">
<h1>HotBox 1.0 - Network & System Monitor</h1>
<div class="card">
  <div class="card-header">Select Network Interface</div>
  <div class="card-body">
    <label for="interface-select">Interface:</label>
    <select id="interface-select" class="form-select"></select>
  </div>
</div>

<div id="dashboard">
  <div class="card" id="metrics-card">
    <div class="card-header">System Metrics</div>
    <div class="card-body">
      <div id="metrics-gauges">
        <div>CPU: <span id="cpu-text">0%</span>
          <div class="progress"><div id="cpu-bar" class="progress-bar" style="width:0%">0%</div></div>
        </div>
        <div>RAM: <span id="ram-text">0%</span>
          <div class="progress"><div id="ram-bar" class="progress-bar" style="width:0%">0%</div></div>
        </div>
        <div>Disk: <span id="disk-text">0%</span>
          <div class="progress"><div id="disk-bar" class="progress-bar" style="width:0%">0%</div></div>
        </div>
        <div>Network RX: <span id="rx-text">0 KB/s</span>
          <div class="progress"><div id="rx-bar" class="progress-bar" style="width:0%">0</div></div>
        </div>
        <div>Network TX: <span id="tx-text">0 KB/s</span>
          <div class="progress"><div id="tx-bar" class="progress-bar" style="width:0%">0</div></div>
        </div>
      </div>
      <ul id="temps-list"></ul>
    </div>
  </div>

  <div class="card" id="clients-card">
    <div class="card-header">Connected WiFi Devices</div>
    <div class="card-body">
      <ul id="client-list"></ul>
    </div>
  </div>
</div>

<script>
const ws = new WebSocket("ws://127.0.0.1:6969/ws");

ws.onmessage = function(event) {
    const data = JSON.parse(event.data);

    // WiFi clients
    const clients = data.wifi_clients || {};
    const clientList = document.getElementById("client-list");
    clientList.innerHTML = "";
    for (const mac in clients) {
        const li = document.createElement("li");
        li.textContent = mac + " — " + clients[mac] + " dBm";
        clientList.appendChild(li);
    }

    // System metrics
    const metricsList = document.getElementById("metrics-list");
    const updateBar = (id, textId, value, maxValue=100, unit='%') => {
        const bar = document.getElementById(id);
        const pct = Math.min(value/maxValue*100, 100);
        bar.style.width = pct + '%';
        bar.textContent = value.toFixed(1) + unit;
        document.getElementById(textId).textContent = value.toFixed(1) + unit;
        bar.style.background = value>85 ? 'linear-gradient(90deg,#ff2222,#ff5555)' :
                           value>60 ? 'linear-gradient(90deg,#ffcc00,#ffaa00)' :
                                      'linear-gradient(90deg,#44ff44,#22cc22)';
    }

    updateBar('cpu-bar','cpu-text', data.cpu_percent);
    updateBar('ram-bar','ram-text', data.ram_percent);
    updateBar('disk-bar','disk-text', data.disk_percent);
    updateBar('rx-bar','rx-text', data.net_rx_rate/1024, 512, ' KB/s');
    updateBar('tx-bar','tx-text', data.net_tx_rate/1024, 512, ' KB/s');

    const tempsList = document.getElementById('temps-list');
    tempsList.innerHTML = "";
    for (const t in data.temps) {
        const temp = data.temps[t];
        const color = temp>80 ? '#ff4444' : temp>60 ? '#ffcc00' : '#44ff44';
        tempsList.innerHTML += '<li>'+t+': <span style="color:'+color+'">'+temp.toFixed(1)+'°C</span></li>';
    }
}

// Populate interface select
fetch("/devices").then(r => r.json()).then(devs => {
    const sel = document.getElementById("interface-select");
    sel.innerHTML = "";
    devs.forEach(d => {
        const opt = document.createElement("option");
        opt.value = d;
        opt.textContent = d;
        sel.appendChild(opt);
    });
    if(devs.length > 0) {
        sel.value = devs[0];
        fetch("/set/" + sel.value);
    }
});

document.getElementById("interface-select").addEventListener("change", function(e) {
    fetch("/set/" + e.target.value);
});
</script>
</body>
</html>
`


// -----------------------------
// MAIN
// -----------------------------
func main() {
	// Detect LAN and WiFi interfaces
	lanIfaces := detectInterfaces()
	wifiIfaces := detectWiFiInterfaces()
	networkDevices = append(lanIfaces, wifiIfaces...)

	if len(networkDevices) > 0 {
		selectedDevice = networkDevices[0]
	}

	ioCounters, _ := net.IOCounters(true)
	for _, c := range ioCounters {
		prevNetIO[c.Name] = c
	}

	go wifiMonitorLoop()
	go openBrowser("http://127.0.0.1:" + PORT)
	server()
}

// -----------------------------
// Detect LAN / Physical interfaces
// -----------------------------
func detectInterfaces() []string {
	devs := []string{}
	ifaces, err := net.Interfaces()
	if err != nil {
		log.Println("interface detect error:", err)
		return devs
	}
	for _, iface := range ifaces {
		name := strings.ToLower(iface.Name)
		if name == "" || strings.Contains(name, "lo") {
			continue
		}
		devs = append(devs, iface.Name)
	}
	log.Println("Detected LAN interfaces:", devs)
	return devs
}

// -----------------------------
// Detect WiFi interfaces using iw
// -----------------------------
func detectWiFiInterfaces() []string {
	wifiIfaces := []string{}
	cmd := exec.Command("iw", "dev")
	out, err := cmd.Output()
	if err != nil {
		log.Println("Error detecting WiFi interfaces:", err)
		return wifiIfaces
	}
	scanner := bufio.NewScanner(strings.NewReader(string(out)))
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		if strings.HasPrefix(line, "Interface") {
			parts := strings.Fields(line)
			if len(parts) >= 2 {
				wifiIfaces = append(wifiIfaces, parts[1])
			}
		}
	}
	log.Println("Detected WiFi interfaces:", wifiIfaces)
	return wifiIfaces
}

// -----------------------------
// Server
// -----------------------------
func server() {
	r := gin.Default()

	r.GET("/", func(c *gin.Context) {
		c.Data(200, "text/html; charset=utf-8", []byte(INDEX_HTML))
	})

	r.GET("/devices", func(c *gin.Context) {
		c.JSON(200, networkDevices)
	})

	r.GET("/set/:dev", func(c *gin.Context) {
		selectedDevice = c.Param("dev")
		log.Println("Selected interface:", selectedDevice)
		c.String(200, "OK")
	})

	r.GET("/ws", func(c *gin.Context) {
		handleWS(c.Writer, c.Request)
	})

	log.Println("HotBox listening on 0.0.0.0:" + PORT)
	r.Run("0.0.0.0:" + PORT)
}

// -----------------------------
// WebSocket
// -----------------------------
func handleWS(w http.ResponseWriter, r *http.Request) {
	conn, err := upgrader.Upgrade(w, r, nil)
	if err != nil {
		log.Println("ws upgrade error:", err)
		return
	}
	defer conn.Close()

	ticker := time.NewTicker(time.Second)
	defer ticker.Stop()

	for range ticker.C {
		metrics := collectMetrics()
		if err := conn.WriteJSON(metrics); err != nil {
			log.Println("ws write error:", err)
			return
		}
	}
}

// -----------------------------
// Collect metrics
// -----------------------------
func collectMetrics() Metrics {
	m := Metrics{
		Timestamp:   time.Now().Unix(),
		Temps:       map[string]float64{},
		WiFiClients: map[string]int{},
	}

	// CPU
	if cpuPercents, err := cpu.Percent(0, false); err == nil && len(cpuPercents) > 0 {
		m.CPUPercent = cpuPercents[0]
	}

	// RAM
	if vm, err := mem.VirtualMemory(); err == nil {
		m.RAMTotal = vm.Total
		m.RAMUsed = vm.Used
		m.RAMPercent = vm.UsedPercent
	}

	// Disk
	if du, err := disk.Usage("/"); err == nil {
		m.DiskTotal = du.Total
		m.DiskUsed = du.Used
		m.DiskPercent = du.UsedPercent
	}

	// Temps
	if temps, err := host.SensorsTemperatures(); err == nil {
		for _, t := range temps {
			if t.Temperature > 0 {
				name := t.SensorKey
				if name == "" {
					name = fmt.Sprintf("sensor-%.2f", t.Temperature)
				}
				m.Temps[name] = t.Temperature
			}
		}
	}

	// Network
	if selectedDevice != "" {
		ioCounters, err := net.IOCounters(true)
		if err == nil {
			for _, c := range ioCounters {
				if c.Name == selectedDevice {
					m.NetRx = c.BytesRecv
					m.NetTx = c.BytesSent

					prevNetLock.Lock()
					prev, ok := prevNetIO[selectedDevice]
					if ok {
						m.NetRxRate = float64(c.BytesRecv-prev.BytesRecv) / 1.0
						m.NetTxRate = float64(c.BytesSent-prev.BytesSent) / 1.0
					}
					prevNetIO[selectedDevice] = c
					prevNetLock.Unlock()
				}
			}
		}
	}

	// WiFi clients
	wifiClientsLock.RLock()
	for k, v := range wifiClients {
		m.WiFiClients[k] = v
	}
	wifiClientsLock.RUnlock()

	return m
}

// -----------------------------
// WiFi monitor
// -----------------------------
func wifiMonitorLoop() {
	for {
		if selectedDevice != "" {
			clients := getSignalStrength(selectedDevice)
			wifiClientsLock.Lock()
			wifiClients = clients
			wifiClientsLock.Unlock()
		}
		time.Sleep(4 * time.Second)
	}
}

// -----------------------------
// Get WiFi signals
// -----------------------------
func getSignalStrength(dev string) map[string]int {
	clients := map[string]int{}
	cmd := exec.Command("iw", "dev", dev, "station", "dump")
	out, err := cmd.Output()
	if err != nil {
		return clients
	}
	scanner := bufio.NewScanner(strings.NewReader(string(out)))
	current := ""
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		if strings.HasPrefix(line, "Station") {
			parts := strings.Fields(line)
			if len(parts) >= 2 {
				current = parts[1]
			} else {
				current = ""
			}
			continue
		}
		if strings.HasPrefix(line, "signal:") && current != "" {
			parts := strings.Fields(line)
			if len(parts) >= 2 {
				if sig, err := strconv.Atoi(parts[1]); err == nil {
					clients[current] = sig
				}
			}
		}
	}
	return clients
}

// -----------------------------
// Auto-open browser
// -----------------------------
func openBrowser(url string) {
	time.Sleep(700 * time.Millisecond)
	switch runtime.GOOS {
	case "linux":
		_ = exec.Command("xdg-open", url).Start()
	case "windows":
		_ = exec.Command("rundll32", "url.dll,FileProtocolHandler", url).Start()
	case "darwin":
		_ = exec.Command("open", url).Start()
	default:
		fmt.Println("Open in browser:", url)
	}
}

