// torque.go
package main

import (
	"bufio"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log"
	"math"
	mathrand "math/rand"
	"net"
	"net/http"
	"os"
	"os/exec"
	"runtime"
	"strconv"
	"strings"
	"sync"
	"time"

	"golang.org/x/net/proxy"
)

/*
Single-file Torque Scanner (Go) - Enhanced UI version 1.1.2 Private Edition - Release.

- Embeds enhanced HTML UI with Tor toggles, jitter/retries/backoff inputs, SSE log streaming
- Optional Tor via SOCKS5 at 127.0.0.1:9050
- Optional Tor NEWNYM via control port at 127.0.0.1:9051
- Concurrency, jitter, retries, exponential backoff
- Loads uas.txt if present; otherwise uses small embedded UA list
- Logs saved to ./logs/tor_scanner_YYYYMMDD_HHMMSS.log

Build:
  go mod init torque
  go mod tidy
  go build -o torque-scanner torque.go
  ./torque-scanner
*/

var (
	embeddedHTML = `<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Torque Scanner - Private Edition</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Orbitron:wght@400;700&display=swap" rel="stylesheet">
<style>
    body {
        background: radial-gradient(circle at center, #0d0d0d 0%, #000000 70%);
        color: #c90000;
        font-family: 'Orbitron', sans-serif;
        letter-spacing: 0.8px;
    }
    h1 { font-weight: 900; letter-spacing: 18px; }
    p { color: lime; }
    .sith-panel { background: rgba(20,0,0,0.75); border:2px solid #8b0000; border-radius:12px; box-shadow:0 0 25px #ff0000aa; padding:20px; }
    .glow-red { text-shadow: 0 0 10px #ff0000; }
    .btn-sith { background: linear-gradient(90deg,#420000,#8b0000); border:none; color:#ffdddd; box-shadow:0 0 12px #ff0000aa; }
    .btn-sith:hover { background: linear-gradient(90deg,#8b0000,#ff0000); color:#fff; box-shadow:0 0 20px #ff0000ff; }
    .scan-terminal { background:#000; color:#ff3333; border:1px solid #8b0000; border-radius:8px; padding:15px; height:320px; overflow-y:auto; font-family:monospace; font-size:0.9rem; box-shadow:inset 0 0 15px #ff0000aa; }
    small { color:#999; }
</style>
</head>
<body>
<div class="container mt-5">
    <h1 class="text-center glow-red mb-4">TORQUE SCANNER</h1>
    <h4 class="text-center text-danger mb-5">Intelligence Probe Console</h4>
    <div class="sith-panel">
        <h3 class="glow-red">⚡ Scanner Control Panel</h3>
        <hr class="border-danger">
        <div class="row mb-3">
            <div class="col-md-6">
                <label class="form-label">Target URL</label>
                <input id="target" class="form-control bg-dark text-danger border-danger" placeholder="http://target.com">
            </div>
            <div class="col-md-3">
                <label class="form-label">Requests</label>
                <input id="count" type="number" class="form-control bg-dark text-danger border-danger" value="10">
            </div>
            <div class="col-md-3">
                <label class="form-label">Concurrency</label>
                <input id="concurrency" type="number" class="form-control bg-dark text-danger border-danger" value="4">
            </div>
        </div>

        <div class="row mb-2">
            <div class="col-md-3">
                <label class="form-label">Min Jitter (s)</label>
                <input id="minj" type="number" class="form-control bg-dark text-danger border-danger" value="1">
            </div>
            <div class="col-md-3">
                <label class="form-label">Max Jitter (s)</label>
                <input id="maxj" type="number" class="form-control bg-dark text-danger border-danger" value="4">
            </div>
            <div class="col-md-3">
                <label class="form-label">Retries</label>
                <input id="retries" type="number" class="form-control bg-dark text-danger border-danger" value="2">
            </div>
            <div class="col-md-3">
                <label class="form-label">Backoff Base</label>
                <input id="backoff" type="number" class="form-control bg-dark text-danger border-danger" value="2">
            </div>
        </div>

        <div class="form-check form-switch my-3">
            <input class="form-check-input" type="checkbox" id="usetor">
            <label class="form-check-label" for="usetor">Use Tor (SOCKS5: localhost:9050)</label>
        </div>
        <div class="form-check form-switch mb-3">
            <input class="form-check-input" type="checkbox" id="newnym">
            <label class="form-check-label" for="newnym">Attempt Tor NEWNYM between requests (control: localhost:9051)</label>
        </div>

        <div class="d-grid gap-2">
            <button id="start" class="btn btn-sith">INITIATE ATTACK PROBE</button>
            <button id="stop" class="btn btn-outline-danger">CEASE ATTACK PROBE</button>
        </div>

        <h4 class="glow-red mt-4">📡 Live Output <small>(SSE)</small></h4>
        <div id="output" class="scan-terminal">
            <p>[System Ready]</p>
            <p>[Awaiting Command...]</p>
        </div>
    </div>
</div>

<script>
let evt;
function appendLine(s){
    const o=document.getElementById("output");
    const p=document.createElement("p");
    p.textContent=s;
    o.appendChild(p);
    o.scrollTop = o.scrollHeight;
}

function startSSE() {
    if (evt) evt.close();
    evt = new EventSource("/events");
    evt.onmessage = function(e){ appendLine(e.data); }
    evt.onerror = function(){ appendLine("[SSE] connection closed"); if(evt) evt.close(); }
}

function stopSSE() {
    if (evt) { evt.close(); evt = null; }
}

document.getElementById("start").addEventListener("click", async function(){
    const target = document.getElementById("target").value;
    const count = parseInt(document.getElementById("count").value||"10");
    const concurrency = parseInt(document.getElementById("concurrency").value||"4");
    const minj = parseFloat(document.getElementById("minj").value||"1");
    const maxj = parseFloat(document.getElementById("maxj").value||"4");
    const retries = parseInt(document.getElementById("retries").value||"2");
    const backoff = parseInt(document.getElementById("backoff").value||"2");
    const usetor = document.getElementById("usetor").checked;
    const newnym = document.getElementById("newnym").checked;

    if(!target){ alert("Please enter a target URL"); return; }
    let t = target;
    if(!/^https?:\/\//i.test(t)){ t = "http://"+t; }

    startSSE();
    appendLine("[Scan starting...]");

    const body = { target: t, count, concurrency, min_jitter: minj, max_jitter: maxj, retries, backoff, use_tor: usetor, tor_newnym: newnym };
    const resp = await fetch("/start", { method:"POST", headers: {"Content-Type":"application/json"}, body: JSON.stringify(body) });
    if(!resp.ok){
        appendLine("[ERROR] Start failed: " + resp.statusText);
    } else {
        appendLine("[Scan started]");
    }
});

document.getElementById("stop").addEventListener("click", async function(){
    appendLine("[Stop requested]");
    await fetch("/stop", { method: "POST" }).catch(()=>{});
    stopSSE();
});
</script>
</body>
</html>
`

	defaultUAs = []string{
		"Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 Chrome/117.0 Safari/537.36",
		"Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 Chrome/117.0 Safari/537.36",
		"Mozilla/5.0 (Macintosh; Intel Mac OS X 13_6) AppleWebKit/605.1.15 Version/16.6 Safari/605.1.15",
		"curl/7.88.1",
	}
)

type StartRequest struct {
	Target      string  `json:"target"`
	Count       int     `json:"count"`
	Concurrency int     `json:"concurrency"`
	MinJitter   float64 `json:"min_jitter"`
	MaxJitter   float64 `json:"max_jitter"`
	Retries     int     `json:"retries"`
	Backoff     int     `json:"backoff"`
	UseTor      bool    `json:"use_tor"`
	TorNewnym   bool    `json:"tor_newnym"`
}

type sseClient struct {
	flusher http.Flusher
	writer  http.ResponseWriter
	mux     sync.Mutex
	closed  bool
}

func (s *sseClient) send(msg string) error {
	s.mux.Lock()
	defer s.mux.Unlock()
	if s.closed {
		return errors.New("closed")
	}
	_, err := fmt.Fprintf(s.writer, "data: %s\n\n", strings.ReplaceAll(msg, "\n", "\\n"))
	if err != nil {
		s.closed = true
		return err
	}
	s.flusher.Flush()
	return nil
}

var (
	sseClients   = make(map[int]*sseClient)
	sseClientsMu sync.Mutex
	nextClientID = 1
	logFile      *os.File
	logger       *log.Logger

	// control channel to stop scans gracefully
	stopScanMu sync.Mutex
	stopScan   bool
)

func addSSEClient(w http.ResponseWriter, r *http.Request) int {
	flusher, ok := w.(http.Flusher)
	if !ok {
		http.Error(w, "Streaming unsupported", http.StatusInternalServerError)
		return 0
	}
	w.Header().Set("Content-Type", "text/event-stream")
	w.Header().Set("Cache-Control", "no-cache")
	w.Header().Set("Connection", "keep-alive")

	client := &sseClient{flusher: flusher, writer: w}
	sseClientsMu.Lock()
	id := nextClientID
	nextClientID++
	sseClients[id] = client
	sseClientsMu.Unlock()

	// use request context to detect disconnect
	ctx := r.Context()
	go func() {
		<-ctx.Done()
		sseClientsMu.Lock()
		if c, ok := sseClients[id]; ok {
			c.closed = true
			delete(sseClients, id)
		}
		sseClientsMu.Unlock()
	}()
	return id
}

func broadcast(msg string) {
	// write to logfile + stdout
	if logger != nil {
		logger.Println(msg)
	}
	sseClientsMu.Lock()
	for id, c := range sseClients {
		if c.closed {
			delete(sseClients, id)
			continue
		}
		_ = c.send(msg)
	}
	sseClientsMu.Unlock()
	fmt.Println(msg)
}

func prepareLogfile() error {
	if err := os.MkdirAll("logs", 0755); err != nil {
		return err
	}
	fname := fmt.Sprintf("logs/torque__%s.log", time.Now().Format("20060102_150405"))
	f, err := os.Create(fname)
	if err != nil {
		return err
	}
	logFile = f
	logger = log.New(io.MultiWriter(f, os.Stdout), "", log.LstdFlags)
	return nil
}

func loadUAs() []string {
	// try to load uas.txt in current dir
	if _, err := os.Stat("user-agents.txt"); err == nil {
		f, err := os.Open("user-agents.txt")
		if err == nil {
			defer f.Close()
			var out []string
			sc := bufio.NewScanner(f)
			for sc.Scan() {
				line := strings.TrimSpace(sc.Text())
				if line != "" {
					out = append(out, line)
				}
			}
			if len(out) > 0 {
				return out
			}
		}
	}
	return defaultUAs
}

func randFloat(min, max float64) float64 {
	return min + mathrand.Float64()*(max-min)
}

func exponentialBackoffSeconds(base int, attempt int) int {
	// base^attempt (attempt starts at 1)
	power := math.Pow(float64(base), float64(attempt))
	return int(power)
}

func maybeDoTorNewnym(host string, port int) {
	addr := fmt.Sprintf("%s:%d", host, port)
	conn, err := net.DialTimeout("tcp", addr, 2*time.Second)
	if err != nil {
		broadcast("[Tor NEWNYM] control port not available: " + err.Error())
		return
	}
	defer conn.Close()
	// simple auth (empty) and SIGNAL NEWNYM, ignore responses
	fmt.Fprintf(conn, "AUTHENTICATE \"\"\r\nSIGNAL NEWNYM\r\nQUIT\r\n")
	conn.SetReadDeadline(time.Now().Add(1 * time.Second))
	buf := make([]byte, 512)
	_, _ = conn.Read(buf)
	broadcast("[Tor NEWNYM] signal sent (if control allowed this)")
}

func makeHTTPClient(useTor bool) *http.Client {
	transport := &http.Transport{
		Proxy: http.ProxyFromEnvironment,
		DialContext: (&net.Dialer{
			Timeout:   15 * time.Second,
			KeepAlive: 30 * time.Second,
		}).DialContext,
		TLSHandshakeTimeout: 10 * time.Second,
	}
	if useTor {
		// attempt SOCKS5 at localhost:9050
		dialer, err := proxy.SOCKS5("tcp", "127.0.0.1:9050", nil, proxy.Direct)
		if err == nil {
			transport.DialContext = func(ctx context.Context, network, addr string) (net.Conn, error) {
				return dialer.Dial(network, addr)
			}
		} else {
			broadcast("[Tor] SOCKS5 dialer creation failed: " + err.Error())
		}
	}
	return &http.Client{Transport: transport, Timeout: 30 * time.Second}
}

func doRequest(client *http.Client, ua string, target string) (string, error) {
	req, err := http.NewRequest("GET", target, nil)
	if err != nil {
		return "", err
	}
	req.Header.Set("User-Agent", ua)
	resp, err := client.Do(req)
	if err != nil {
		return "", err
	}
	defer resp.Body.Close()
	return strconv.Itoa(resp.StatusCode), nil
}

func worker(id int, jobs <-chan string, wg *sync.WaitGroup, client *http.Client, uas []string, retries int, backoff int, minj, maxj float64, useTor bool, torNewnym bool) {
	defer wg.Done()
	r := mathrand.New(mathrand.NewSource(time.Now().UnixNano() + int64(id)))
	for job := range jobs {
		// check for global stop signal
		stopScanMu.Lock()
		if stopScan {
			stopScanMu.Unlock()
			broadcast(fmt.Sprintf("[Worker %d] stopping early", id))
			return
		}
		stopScanMu.Unlock()

		ua := uas[r.Intn(len(uas))]
		jitter := randFloat(minj, maxj)
		time.Sleep(time.Duration(jitter * float64(time.Second)))

		var status string
		var err error
		attempt := 0
		for attempt = 1; attempt <= retries+1; attempt++ {
			// if Tor NEWNYM requested, signal between attempts (attempt >1)
			if useTor && torNewnym && attempt > 1 {
				maybeDoTorNewnym("127.0.0.1", 9051)
			}

			// perform request
			status, err = doRequest(client, ua, job)
			// treat 2xx/3xx/4xx as success (matching original bash behavior)
			if err == nil && (strings.HasPrefix(status, "2") || strings.HasPrefix(status, "3") || strings.HasPrefix(status, "4")) {
				break
			}
			if attempt <= retries {
				waitSec := exponentialBackoffSeconds(backoff, attempt)
				broadcast(fmt.Sprintf("[Worker %d] Attempt %d failed for %s (%v). Retrying in %ds", id, attempt, job, err, waitSec))
				time.Sleep(time.Duration(waitSec) * time.Second)
			}
		}

		if err != nil {
			broadcast(fmt.Sprintf("[Worker %d] %s -> ERROR: %v (attempts=%d)", id, job, err, attempt-1))
			continue
		}

		// --- Output formatting ---
		torNote := ""
		if useTor {
			torNote = " (via tor)"
		}

		broadcast(fmt.Sprintf(
			"[Worker %d] [Request]%s %s Status: %s UA: %s attempts=%d",
			id,
			torNote,
			job,
			status,
			ua,
			attempt,
		))
	}
}

func startScan(req StartRequest) error {
	// validation & defaults
	if req.Target == "" {
		return errors.New("target required")
	}
	if req.Count < 1 {
		req.Count = 1
	}
	if req.Concurrency < 1 {
		req.Concurrency = 1
	}
	if req.MinJitter < 0 {
		req.MinJitter = 0
	}
	if req.MaxJitter < req.MinJitter {
		req.MaxJitter = req.MinJitter
	}
	if req.Retries < 0 {
		req.Retries = 0
	}
	if req.Backoff < 1 {
		req.Backoff = 2
	}

	// prepare logfile once
	if logger == nil {
		if err := prepareLogfile(); err != nil {
			return err
		}
	}

	// reset global stop flag
	stopScanMu.Lock()
	stopScan = false
	stopScanMu.Unlock()

	broadcast(fmt.Sprintf("[Scan] target=%s count=%d concurrency=%d minj=%.2f maxj=%.2f retries=%d backoff=%d use_tor=%v tor_newnym=%v",
		req.Target, req.Count, req.Concurrency, req.MinJitter, req.MaxJitter, req.Retries, req.Backoff, req.UseTor, req.TorNewnym))

	uas := loadUAs()
	client := makeHTTPClient(req.UseTor)

	// queue
	jobs := make(chan string, req.Count)
	for i := 0; i < req.Count; i++ {
		jobs <- req.Target
	}
	close(jobs)

	var wg sync.WaitGroup
	workers := req.Concurrency
	if workers > req.Count {
		workers = req.Count
	}
	wg.Add(workers)
	for i := 0; i < workers; i++ {
		go worker(i+1, jobs, &wg, client, uas, req.Retries, req.Backoff, req.MinJitter, req.MaxJitter, req.UseTor, req.TorNewnym)
	}
	wg.Wait()
	broadcast("[Scan] complete")
	return nil
}

func handlerRoot(w http.ResponseWriter, r *http.Request) {
	w.Write([]byte(embeddedHTML))
}

func handlerStart(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		http.Error(w, "method", http.StatusMethodNotAllowed)
		return
	}
	var sr StartRequest
	if err := json.NewDecoder(r.Body).Decode(&sr); err != nil {
		http.Error(w, "bad request: "+err.Error(), http.StatusBadRequest)
		return
	}
	// kick off scan in background
	go func() {
		if err := startScan(sr); err != nil {
			broadcast("[ERROR] " + err.Error())
		}
	}()
	w.WriteHeader(http.StatusAccepted)
	w.Write([]byte("started"))
}

func handlerStop(w http.ResponseWriter, r *http.Request) {
	// set global stop flag
	stopScanMu.Lock()
	stopScan = true
	stopScanMu.Unlock()
	broadcast("[STOP] graceful stop requested")
	w.WriteHeader(http.StatusAccepted)
	w.Write([]byte("stopping"))
}

func handlerEvents(w http.ResponseWriter, r *http.Request) {
	id := addSSEClient(w, r)
	if id == 0 {
		return
	}
	// keep connection open; the addSSEClient goroutine will remove client on context done
	<-r.Context().Done()
}

func openBrowser(url string) {
	var cmd *exec.Cmd
	switch runtime.GOOS {
	case "darwin":
		cmd = exec.Command("open", url)
	case "windows":
		cmd = exec.Command("rundll32", "url.dll,FileProtocolHandler", url)
	default:
		cmd = exec.Command("xdg-open", url)
	}
	_ = cmd.Start()
}

func main() {
	mathrand.Seed(time.Now().UnixNano())
	_ = loadUAs()

	mux := http.NewServeMux()
	mux.HandleFunc("/", handlerRoot)
	mux.HandleFunc("/start", handlerStart)
	mux.HandleFunc("/stop", handlerStop)
	mux.HandleFunc("/events", handlerEvents)

	server := &http.Server{
		Addr:    "127.0.0.1:0", // random free port
		Handler: mux,
	}
	ln, err := net.Listen("tcp", server.Addr)
	if err != nil {
		fmt.Println("Failed to bind:", err)
		return
	}
	actualAddr := ln.Addr().String()
	uiURL := "http://" + actualAddr + "/"
	fmt.Println("Serving UI on", uiURL)
	// open browser (best effort)
	go func() {
		time.Sleep(200 * time.Millisecond)
		openBrowser(uiURL)
	}()
	if err := server.Serve(ln); err != nil && err != http.ErrServerClosed {
		fmt.Println("Server error:", err)
	}
}

