#!/usr/bin/env bash
set -euo pipefail

# build_holocron.sh
# Builds holocron.py into a single-file executable for the CURRENT platform,
# installs dependencies into a temporary venv, uses PyInstaller, and places
# asn.db next to the final executable (external, so you can update it later).
#
# Usage:
#   ./build_holocron.sh
#
# Run this script on the OS you want to build for (Linux, macOS, Windows via Git Bash).
# For multi-platform builds, use the included GitHub Actions workflow.

APP_NAME="holocron"
MAIN="holocron.py"
DB_FILE="asn.db"
VENV_DIR=".build_env"
DIST_ROOT="release"
BUILD_LOG="build.log"

echo "=== Holocron Builder ==="
echo "Main script: ${MAIN}"
echo "Database file: ${DB_FILE}"
echo

# Check files
if [ ! -f "${MAIN}" ]; then
    echo "ERROR: ${MAIN} not found in current directory."
    exit 2
fi

if [ ! -f "${DB_FILE}" ]; then
    echo "WARNING: ${DB_FILE} not found. The built binary will still be placed in dist/ but DB won't be copied."
fi

# Python check
if ! command -v python3 >/dev/null 2>&1; then
    echo "ERROR: python3 not found. Install Python 3.8+"
    exit 3
fi

PYTHON_BIN=$(command -v python3)
echo "Using Python at: ${PYTHON_BIN}"

# Remove previous build artifacts
echo "Cleaning previous builds..."
rm -rf "${VENV_DIR}" build dist "${DIST_ROOT}" "${BUILD_LOG}"

# Create venv
echo "Creating virtualenv..."
python3 -m venv "${VENV_DIR}"
# shellcheck disable=SC1091
source "${VENV_DIR}/bin/activate"

# Upgrade pip
python -m pip install --upgrade pip wheel setuptools >/dev/null

# Install runtime requirements + pyinstaller
echo "Installing requirements into venv..."
pip install flask requests >/dev/null
pip install pyinstaller >/dev/null

# Optional: create a requirements.txt snapshot for reproducibility
pip freeze > ./${VENV_DIR}/requirements-freeze.txt

# PyInstaller build options
# Use --onefile so we produce a single binary. We keep DB external (not bundled).
PYINSTALLER_NAME="${APP_NAME}"
if [[ "$(uname -s)" == *"MINGW"* || "$(uname -s)" == *"MSYS"* || "$(uname -s)" == *"CYGWIN"* ]]; then
    # On Windows/Git Bash produce .exe naming
    PYINSTALLER_NAME="${APP_NAME}.exe"
fi

echo "Building ${PYINSTALLER_NAME} with PyInstaller..."
# Use --noconfirm to avoid interactive overwrites
pyinstaller --onefile --noconfirm --name "${APP_NAME}" "${MAIN}" > "${BUILD_LOG}" 2>&1

echo "PyInstaller finished. See ${BUILD_LOG} for details."

# Prepare release folder for this platform
mkdir -p "${DIST_ROOT}"
PLATFORM="$(uname -s)"
OUTPUT_DIR="${DIST_ROOT}/${APP_NAME}_${PLATFORM}"
rm -rf "${OUTPUT_DIR}"
mkdir -p "${OUTPUT_DIR}"

# Move the produced file from dist/
if [[ -f "dist/${APP_NAME}" ]]; then
    mv "dist/${APP_NAME}" "${OUTPUT_DIR}/"
elif [[ -f "dist/${APP_NAME}.exe" ]]; then
    mv "dist/${APP_NAME}.exe" "${OUTPUT_DIR}/"
else
    echo "ERROR: expected dist/${APP_NAME} or dist/${APP_NAME}.exe not found."
    echo "Contents of dist/:"
    ls -la dist || true
    deactivate || true
    exit 6
fi

# Copy asn.db next to the executable (external file)
if [ -f "${DB_FILE}" ]; then
    cp "${DB_FILE}" "${OUTPUT_DIR}/"
    echo "Copied ${DB_FILE} into ${OUTPUT_DIR}/"
else
    echo "asn.db not present; skipping copy. You can place it in ${OUTPUT_DIR}/ later."
fi

# Copy any other runtime assets (if you have templates/static) - optional
# cp -r templates "${OUTPUT_DIR}/" || true
# cp -r static "${OUTPUT_DIR}/" || true

# Create a zip/tar for distribution
ARCHIVE_NAME="${APP_NAME}_${PLATFORM}.zip"
pushd "${DIST_ROOT}" >/dev/null
zip -r "../${ARCHIVE_NAME}" "${APP_NAME}_${PLATFORM}" >/dev/null
popd >/dev/null

echo "Built and packaged: ${ARCHIVE_NAME}"
echo "Final output folder: ${OUTPUT_DIR}"
echo

# Cleanup intermediate pyinstaller directories but keep the dist release
rm -rf build __pycache__ "${VENV_DIR}" dist "${APP_NAME}.spec"

echo "Build complete."
echo "If you want Windows or macOS binaries and are not on those platforms,"
echo "either run this script on those platforms or use the provided GitHub Actions workflow (recommended)."

