# HOLOCRON 2.0 holocron.py
# Network Intelligence Console
# K0NxT3D 2025

from flask import Flask, render_template_string, request, Response, jsonify
import sqlite3
import webbrowser
from threading import Timer
import csv
import io
import requests

app = Flask('Holocron')
DB_FILE = "asn.db"
PORT = 35001

# =====================================================
# Templates
# =====================================================

HTML_TEMPLATE = """
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8"/>
<meta name="viewport" content="width=device-width,initial-scale=1"/>
<title>Holocron Console v2.0 — K0NxT3D</title>

<!-- Bootstrap CSS -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">

<!-- Sith Fonts -->
<link href="https://fonts.googleapis.com/css2?family=Orbitron:wght@400;700&family=Cinzel:wght@400;700&display=swap" rel="stylesheet">

<style>
    :root {
        --sith-red: #990000;
        --dark-bg: #050000;
        --dark-panel: #0a0000;
        --ancient-gold: #d4b56a;
        --text-main: #d9d3d3;
        --scanline: rgba(255,0,0,0.05);
    }

    html, body {
        background: radial-gradient(ellipse at center, var(--dark-bg) 0%, #000 80%);
        color: var(--text-main);
        font-family: 'Orbitron', 'Cinzel', monospace;
        height: 100%;
        overflow-x: hidden;
        position: relative;
    }

    /* Animated scanlines overlay */
    body::before {
        content: "";
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: repeating-linear-gradient(
            to bottom,
            var(--scanline) 0px,
            var(--scanline) 1px,
            transparent 2px,
            transparent 3px
        );
        animation: flicker 1.8s infinite alternate;
        pointer-events: none;
        z-index: 1;
    }

    @keyframes flicker {
        0% { opacity: 0.15; }
        100% { opacity: 0.25; }
    }

    .panel {
        background: linear-gradient(180deg, rgba(15,0,0,0.8), rgba(0,0,0,0.9));
        border: 1px solid rgba(153,0,0,0.6);
        box-shadow: 0 0 25px rgba(153,0,0,0.3);
        border-radius: 10px;
        padding: 20px;
        margin-top: 20px;
        position: relative;
        z-index: 2;
    }

    h1 {
        color: var(--ancient-gold);
        text-shadow: 0 0 15px rgba(153,0,0,0.8);
        letter-spacing: 3px;
        font-weight: 900;
    }

    .subhead {
        color: #aa8888;
        font-size: 0.9rem;
    }

    .form-control, .form-select {
        background: rgba(0,0,0,0.6);
        border: 1px solid rgba(153,0,0,0.5);
        color: #e6e6e6;
    }

    .form-control:focus, .form-select:focus {
        border-color: var(--sith-red);
        box-shadow: 0 0 6px rgba(153,0,0,0.6);
    }

    .btn-primary {
        background: linear-gradient(180deg, #990000, #330000);
        border: 1px solid #550000;
        color: #f2dede;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    .btn-primary:hover {
        background: linear-gradient(180deg, #bb0000, #440000);
        box-shadow: 0 0 12px rgba(255,0,0,0.4);
    }

    .btn-ghost {
        border: 1px solid rgba(153,0,0,0.4);
        color: var(--ancient-gold);
        background: transparent;
    }

    /* Sith-style Table */
    .table {
        background: rgba(20,0,0,0.85);
        border-radius: 6px;
        color: #f0e6e6;
    }

    .table thead th {
        background: rgba(153,0,0,0.4);
        color: var(--ancient-gold);
        border-bottom: 1px solid rgba(153,0,0,0.6);
    }

    .table tbody tr {
        background: rgba(0,0,0,0.5);
        border-bottom: 1px solid rgba(153,0,0,0.2);
    }

    .table tbody tr:hover {
        background: rgba(153,0,0,0.2);
        transition: background 0.3s ease-in-out;
    }

    .table td {
background-color: #111;
        color: #e0e0e0 !important;
    }

    /* Modal */
    .modal-content {
        background: linear-gradient(180deg, #0a0000 0%, #000000 100%);
        border: 1px solid rgba(153,0,0,0.5);
        color: #f0e0e0;
        animation: pulseModal 2s infinite;
    }

    @keyframes pulseModal {
        0%, 100% { box-shadow: 0 0 12px rgba(153,0,0,0.4); }
        50% { box-shadow: 0 0 24px rgba(153,0,0,0.8); }
    }

    .modal-title {
        color: var(--ancient-gold);
    }

    .prefix-badge {
        display: inline-block;
        background: rgba(153,0,0,0.3);
        border: 1px solid rgba(153,0,0,0.6);
        color: #f0e0e0;
        border-radius: 5px;
        padding: 6px 10px;
        margin: 3px;
        font-family: 'Orbitron', monospace;
    }

    footer.small {
        margin-top: 20px;
        color: #883;
        font-size: 0.8rem;
        text-align: center;
    }

    /* Console log effect (optional aesthetic panel) */
    #console-log {
        background: rgba(10,0,0,0.5);
        border: 1px solid rgba(153,0,0,0.4);
        padding: 10px;
        margin-top: 20px;
        height: 120px;
        overflow-y: auto;
        font-family: monospace;
        font-size: 0.85rem;
        color: #d4b56a;
        box-shadow: inset 0 0 10px rgba(153,0,0,0.2);
    }

    #console-log span {
        display: block;
        opacity: 0.9;
    }
</style>
</head>
<body>
<div class="container">
    <div class="panel">
        <div class="d-flex justify-content-between align-items-center mb-2">
            <div>
                <h1>HOLOCRON</h1>
                <div class="subhead">Holocron Data Archive Interface</div>
            </div>
            <button class="btn btn-ghost" id="theme-hint">SYSTEM MODE: ARCHIVE</button>
        </div>

        <form method="POST" class="search-form row g-3 align-items-center">
            <div class="col-md-6">
                <input type="text" class="form-control" name="search_term" placeholder="Enter search term (ASN, name, location...)" value="{{ search_term }}">
            </div>
            <div class="col-md-3">
                <select class="form-select" name="search_by">
                    <option value="ASN" {% if search_by == "ASN" %}selected{% endif %}>ASN</option>
                    <option value="Name" {% if search_by == "Name" %}selected{% endif %}>Name</option>
                    <option value="Type" {% if search_by == "Type" %}selected{% endif %}>Type</option>
                    <option value="Location" {% if search_by == "Location" %}selected{% endif %}>Location</option>
                </select>
            </div>
            <div class="col-md-3">
                <button type="submit" class="btn btn-primary w-100">SEARCH</button>
            </div>
        </form>

        {% if results is not none %}
        <hr style="border-color: rgba(153,0,0,0.3)">
        <h5 style="color:#bb9999;">Results</h5>
        <div class="table-responsive">
            <table class="table table-borderless">
                <thead>
                    <tr>
                        <th>ASN</th>
                        <th>Name</th>
                        <th>Type</th>
                        <th>Location</th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    {% for result in results %}
                    <tr>
                        <td>{{ result[0] }}</td>
                        <td>{{ result[1] }}</td>
                        <td>{{ result[2] }}</td>
                        <td>{{ result[3] }}</td>
                        <td><button class="btn btn-primary btn-sm" onclick="openRangesModal('{{ result[0] }}')">Ranges</button></td>
                    </tr>
                    {% else %}
                    <tr><td colspan="5" class="text-center">No results found.</td></tr>
                    {% endfor %}
                </tbody>
            </table>
        </div>

        <form method="POST" action="/export_csv" class="mt-3">
            <input type="hidden" name="search_term" value="{{ search_term }}">
            <input type="hidden" name="search_by" value="{{ search_by }}">
            <button type="submit" class="btn btn-primary">Export Table CSV</button>
        </form>
        {% endif %}

        <div id="console-log">
            <span>[INIT] Holocron systems active.</span>
            <span>[READY] Awaiting command input...</span>
        </div>

        <footer class="small">Holocron Console v2.0 — <span style="color:var(--ancient-gold)">K0NxT3D</span></footer>
    </div>
</div>

<!-- Modal -->
<div class="modal fade" id="rangesModal" tabindex="-1" aria-labelledby="rangesModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="rangesModalLabel">IP Ranges</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div id="ranges-loading">Querying the database archives...</div>
        <div id="ranges-list" class="mt-3"></div>
        <div id="ranges-empty" class="text-muted" style="display:none">No data found.</div>

        <form id="exportRangesForm" method="POST" action="/export_ranges_csv" target="_blank">
            <input type="hidden" name="asn" id="export-asn">
            <button type="submit" class="btn btn-primary mt-3">Export Ranges CSV</button>
        </form>
      </div>
    </div>
  </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
const modalEl = document.getElementById('rangesModal');
const bsModal = new bootstrap.Modal(modalEl, { keyboard: true });
const consoleLog = document.getElementById('console-log');

function logConsole(msg) {
    const span = document.createElement('span');
    span.textContent = msg;
    consoleLog.appendChild(span);
    consoleLog.scrollTop = consoleLog.scrollHeight;
}

function openRangesModal(asn) {
    document.getElementById('ranges-loading').style.display = 'block';
    document.getElementById('ranges-list').innerHTML = '';
    document.getElementById('ranges-empty').style.display = 'none';
    document.getElementById('export-asn').value = asn;
    bsModal.show();

    let normalized = asn.toString().trim();
    if (!/^AS/i.test(normalized)) normalized = 'AS' + normalized;

    fetch('/ranges_json/' + encodeURIComponent(normalized))
        .then(r => r.json())
        .then(data => {
            document.getElementById('ranges-loading').style.display = 'none';
            const list = document.getElementById('ranges-list');
            if (!data.prefixes || !data.prefixes.length) {
                document.getElementById('ranges-empty').style.display = 'block';
                logConsole('[WARN] No ranges found for ' + normalized);
                return;
            }
            data.prefixes.forEach(p => {
                const el = document.createElement('span');
                el.className = 'prefix-badge';
                el.textContent = p;
                list.appendChild(el);
            });
            logConsole('[INFO] ' + data.prefixes.length + ' prefixes retrieved for ' + normalized);
        })
        .catch(err => {
            console.error(err);
            document.getElementById('ranges-loading').textContent = 'Error fetching data.';
            logConsole('[ERROR] Failed to retrieve ranges for ' + asn);
        });
}
</script>
</body>
</html>
"""


# =====================================================
# Database and API Logic
# =====================================================

def query_database(search_term, search_by):
    conn = sqlite3.connect(DB_FILE)
    cursor = conn.cursor()

    if search_by == "ASN":
        query = "SELECT * FROM asn WHERE asn LIKE ?"
    elif search_by == "Name":
        query = "SELECT * FROM asn WHERE name LIKE ?"
    elif search_by == "Type":
        query = "SELECT * FROM asn WHERE type LIKE ?"
    elif search_by == "Location":
        query = "SELECT * FROM asn WHERE location LIKE ?"
    else:
        return []

    cursor.execute(query, (f"%{search_term}%",))
    results = cursor.fetchall()
    conn.close()
    return results

def fetch_prefixes(asn):
    """Fetch IP prefixes for an ASN using the RIPE Stat API."""
    try:
        url = f"https://stat.ripe.net/data/announced-prefixes/data.json?resource={asn}"
        response = requests.get(url, timeout=12)
        if response.status_code == 200:
            data = response.json()
            prefixes = [entry["prefix"] for entry in data.get("data", {}).get("prefixes", [])]
            return prefixes
        return []
    except Exception as e:
        print(f"Error fetching prefixes for {asn}: {e}")
        return []

# =====================================================
# Routes
# =====================================================

@app.route("/", methods=["GET", "POST"])
def home():
    if request.method == "POST":
        search_term = request.form.get("search_term", "").strip()
        search_by = request.form.get("search_by", "ASN")
        results = query_database(search_term, search_by)
        return render_template_string(HTML_TEMPLATE, results=results, search_term=search_term, search_by=search_by)
    return render_template_string(HTML_TEMPLATE, results=None, search_term="", search_by="ASN")

@app.route("/ranges/<asn>")
def show_ranges(asn):
    # Legacy route — keep for direct linking; returns a simple page
    prefixes = fetch_prefixes(asn)
    # present a simple page in case the user opens directly (not used by modal)
    html = "<h1>Prefixes for {}</h1><ul>{}</ul><a href='/'>Back</a>".format(
        asn, "".join(f"<li>{p}</li>" for p in prefixes)
    )
    return html

@app.route("/ranges_json/<asn>")
def ranges_json(asn):
    """Return prefixes as JSON for the modal to consume."""
    prefixes = fetch_prefixes(asn)
    return jsonify({"asn": asn, "prefixes": prefixes})

@app.route("/export_csv", methods=["POST"])
def export_csv():
    search_term = request.form.get("search_term", "").strip()
    search_by = request.form.get("search_by", "ASN")
    results = query_database(search_term, search_by)

    output = io.StringIO()
    writer = csv.writer(output)
    writer.writerow(["ASN", "Name", "Type", "Location"])
    for row in results:
        writer.writerow(row)
    output.seek(0)
    return Response(output, mimetype="text/csv",
                    headers={"Content-Disposition": "attachment;filename=asn_results.csv"})

@app.route("/export_ranges_csv", methods=["POST"])
def export_ranges_csv():
    asn = request.form.get("asn", "").strip()
    # normalize ASN to AS123 style
    if asn and not asn.upper().startswith("AS"):
        asn = "AS" + asn
    prefixes = fetch_prefixes(asn)

    output = io.StringIO()
    writer = csv.writer(output)
    writer.writerow(["ASN", "Prefix"])
    for prefix in prefixes:
        writer.writerow([asn, prefix])
    output.seek(0)
    return Response(output, mimetype="text/csv",
                    headers={"Content-Disposition": f"attachment;filename={asn}_prefixes.csv"})

# =====================================================
# Run the app
# =====================================================

def open_browser():
    webbrowser.open(f"http://127.0.0.1:{PORT}")

if __name__ == "__main__":
    Timer(1, open_browser).start()
    app.run(debug=True, port=PORT, use_reloader=False)

