#!/usr/bin/env python3
# Title: ReconX - Domain Reconnaissance Spyglass
# Author: K0NxT3D
# Description: Retrieve useful data related to a given domain.
# URL: http://www.seaverns.com/security/reconx-domain-reconnaissance-spyglass/
# File: reconx.py

import dns.resolver
import os
import socket
import ssl
import threading
import queue
from datetime import datetime, timezone
import sys
import html
import traceback

# ========= PATCH: FIX UBUNTU 24 DNS RESOLUTION ==========
resolver = dns.resolver.Resolver()
resolver.nameservers = ["8.8.8.8", "1.1.1.1", "9.9.9.9"]
# ========================================================

# ========= CONFIGURATION ==========
SUBDOMAIN_THREADS = 20
PORT_THREADS = 20
BRUTE_THREADS = 50
REPORTS_DIR = "reconx_reports"
if not os.path.isdir(REPORTS_DIR):
    os.makedirs(REPORTS_DIR, exist_ok=True)

# ========= COLOR SYSTEM ==========
class C:
    RED = "\033[91m"
    GREEN = "\033[92m"
    YELLOW = "\033[93m"
    BLUE = "\033[94m"
    CYAN = "\033[96m"
    MAG = "\033[95m"
    W = "\033[97m"
    RST = "\033[0m"

# ========= OUTPUT BUFFER ==========
output_buffer = []
lock_output = threading.Lock()

def log(msg, store=True):
    """Thread-safe print + optional store for report."""
    with lock_output:
        print(msg)
        if store:
            output_buffer.append(msg)

# ========= BANNER ==========
BANNER = f"""
{C.YELLOW}                                          
@@@@@@@   @@@@@@@@   @@@@@@@   @@@@@@   @@@  @@@  @@@  @@@  
@@@@@@@@  @@@@@@@@  @@@@@@@@  @@@@@@@@  @@@@ @@@  @@@  @@@  
@@!  @@@  @@!       !@@       @@!  @@@  @@!@!@@@  @@!  !@@  
!@!  @!@  !@!       !@!       !@!  @!@  !@!!@!@!  !@!  @!!  
@!@!!@!   @!!!:!    !@!       @!@  !@!  @!@ !!@!   !@@!@!   
!!@!@!    !!!!!:    !!!       !@!  !!!  !@!  !!!    @!!!    
!!: :!!   !!:       :!!       !!:  !!!  !!:  !!!   !: :!!   
:!:  !:!  :!:       :!:       :!:  !:!  :!:  !:!  :!:  !:!  
::   :::   :: ::::   ::: :::  ::::: ::   ::   ::   ::  :::  
 :   : :  : :: ::    :: :: :   : :  :   ::    :    :   ::   
     
{C.CYAN}ReconX - Domain Reconnaissance Spyglass - Build V2.1{C.RST}
"""

# ========= COMMON SUBDOMAINS (default wordlist) ==========
DEFAULT_SUBS = [
    "www","mail","blog","shop","store","webmail","login","members","news","admin","ftp","api","dev","staging",
    "test","secure","portal","cpanel","direct","static","cdn","ns1","ns2","mx","smtp","web","beta","monitor",
    "git","gitlab","jira","gitweb","images","img","assets","mobile","m","owa","owa2","vhost","vpn","dashboard"
]

# ========= SUBDOMAIN SCAN (THREAD POOL) ==========
def worker_subdomain(domain, q, results):
    while True:
        sub = q.get()
        if sub is None:
            q.task_done()
            break
        full_domain = f"{sub}.{domain}"
        try:
            answers = resolver.resolve(full_domain, 'A')
            for a in answers:
                results_lock.acquire()
                results.append((full_domain, a.to_text()))
                results_lock.release()
        except Exception:
            # ignore quietly
            pass
        q.task_done()

def check_subdomains(domain, sublist=None, thread_count=SUBDOMAIN_THREADS):
    if sublist is None:
        sublist = DEFAULT_SUBS
    q = queue.Queue()
    results = []
    global results_lock
    results_lock = threading.Lock()
    threads = []
    for _ in range(thread_count):
        t = threading.Thread(target=worker_subdomain, args=(domain, q, results), daemon=True)
        t.start()
        threads.append(t)
    for s in sublist:
        q.put(s.strip())
    q.join()
    for _ in threads:
        q.put(None)
    for t in threads:
        t.join()
    # remove duplicates and sort
    uniq = {}
    for d, ip in results:
        uniq[d] = ip
    return sorted(uniq.items())

# ========= BRUTE FORCE SUBDOMAIN ==========
def worker_brute(domain, q, results):
    while True:
        w = q.get()
        if w is None:
            q.task_done()
            break
        candidate = f"{w}.{domain}"
        try:
            answers = resolver.resolve(candidate, 'A')
            ips = [a.to_text() for a in answers]
            results_lock.acquire()
            results.append((candidate, ips))
            results_lock.release()
        except Exception:
            pass
        q.task_done()

def brute_subdomains(domain, wordlist_path, thread_count=BRUTE_THREADS):
    if not os.path.isfile(wordlist_path):
        raise FileNotFoundError(f"Wordlist not found: {wordlist_path}")
    q = queue.Queue()
    results = []
    global results_lock
    results_lock = threading.Lock()
    threads = []
    for _ in range(thread_count):
        t = threading.Thread(target=worker_brute, args=(domain, q, results), daemon=True)
        t.start()
        threads.append(t)
    with open(wordlist_path, "r", errors="ignore") as fh:
        for line in fh:
            word = line.strip()
            if word:
                q.put(word)
    q.join()
    for _ in threads:
        q.put(None)
    for t in threads:
        t.join()
    return sorted(results)

# ========= SSL CERTIFICATE ==========
def get_ssl_certificate(domain):
    try:
        context = ssl.create_default_context()
        conn = context.wrap_socket(socket.socket(socket.AF_INET), server_hostname=domain)
        conn.settimeout(5)
        conn.connect((domain, 443))
        cert = conn.getpeercert()
        conn.close()
        return cert
    except Exception as e:
        return {"error": str(e)}

# ========= PORT SCANNER (THREAD PER PORT) ==========
COMMON_PORTS = {
    21: "FTP", 22: "SSH", 23: "TELNET", 25: "SMTP", 53: "DNS",
    80: "HTTP", 110: "POP3", 143: "IMAP", 443: "HTTPS", 587: "SMTP (TLS)",
    3306: "MySQL", 8080: "HTTP-ALT"
}

def worker_port_scan(ip, q, results):
    while True:
        item = q.get()
        if item is None:
            q.task_done()
            break
        port, service = item
        try:
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(0.6)
            res = sock.connect_ex((ip, port))
            if res == 0:
                results_lock.acquire()
                results.append((port, service))
                results_lock.release()
            sock.close()
        except Exception:
            pass
        q.task_done()

def scan_ports(domain, thread_count=PORT_THREADS):
    try:
        ip = socket.gethostbyname(domain)
    except Exception as e:
        return {"error": f"Could not resolve domain: {e}"}
    q = queue.Queue()
    results = []
    global results_lock
    results_lock = threading.Lock()
    threads = []
    for _ in range(min(thread_count, len(COMMON_PORTS))):
        t = threading.Thread(target=worker_port_scan, args=(ip, q, results), daemon=True)
        t.start()
        threads.append(t)
    for port, svc in COMMON_PORTS.items():
        q.put((port, svc))
    q.join()
    for _ in threads:
        q.put(None)
    for t in threads:
        t.join()
    return {"ip": ip, "open_ports": sorted(results)}

# ========= REPORT WRITERS ==========
def write_text_report(domain, data):
    safe_domain = domain.replace(".", "_")
    ts = datetime.now(timezone.utc).strftime("%Y%m%d_%H%M%S")
    filename = os.path.join(REPORTS_DIR, f"reconx_{safe_domain}_{ts}.txt")
    with open(filename, "w", encoding="utf-8") as f:
        f.write("\n".join(output_buffer))
    return filename

def write_html_report(domain, data):
    safe_domain = domain.replace(".", "_")
    ts = datetime.now(timezone.utc).strftime("%Y%m%d_%H%M%S")
    filename = os.path.join(REPORTS_DIR, f"reconx_{safe_domain}_{ts}.html")
    title = f"ReconX Report - {domain} - {ts}"
    css = """
    body { background: #0b0b0c; color: #e6e6e6; font-family: monospace; padding: 18px; }
    .card { background:#0f0f10; border:1px solid #222; padding:12px; margin-bottom:12px; border-radius:6px; box-shadow: 0 4px 18px rgba(0,0,0,0.6);}
    h1 { color: #f2a900; }
    table { width:100%; border-collapse:collapse; }
    th,td { border-bottom:1px solid #222; padding:6px; text-align:left; font-size:13px; }
    .muted { color:#9aa0a6; font-size:12px; }
    """
    # HTML-escape content
    def esc(s):
        return html.escape(str(s))
    with open(filename, "w", encoding="utf-8") as f:
        f.write(f"<!doctype html><html><head><meta charset='utf-8'><title>{esc(title)}</title><style>{css}</style></head><body>")
        f.write(f"<h1>{esc(title)}</h1>")
        f.write(f"<div class='muted'>Generated: {esc(datetime.now(timezone.utc).isoformat())} UTC</div>")
        # Subdomains
        f.write("<div class='card'><h2>Subdomains</h2>")
        subs = data.get("subdomains", [])
        if subs:
            f.write("<table><tr><th>Hostname</th><th>IP(s)</th></tr>")
            for h, ip in subs:
                f.write(f"<tr><td>{esc(h)}</td><td>{esc(ip)}</td></tr>")
            f.write("</table>")
        else:
            f.write("<div class='muted'>No subdomains found.</div>")
        f.write("</div>")
        # Brute results
        brute = data.get("brute", [])
        f.write("<div class='card'><h2>Brute-force Results</h2>")
        if brute:
            f.write("<table><tr><th>Hostname</th><th>IP(s)</th></tr>")
            for h, ips in brute:
                f.write(f"<tr><td>{esc(h)}</td><td>{esc(', '.join(ips))}</td></tr>")
            f.write("</table>")
        else:
            f.write("<div class='muted'>No brute-forced subdomains found or none attempted.</div>")
        f.write("</div>")
        # SSL
        sslinfo = data.get("ssl", {})
        f.write("<div class='card'><h2>SSL Certificate</h2>")
        if isinstance(sslinfo, dict) and "error" not in sslinfo:
            f.write("<table>")
            for k, v in sslinfo.items():
                f.write(f"<tr><th>{esc(k)}</th><td>{esc(v)}</td></tr>")
            f.write("</table>")
        else:
            f.write(f"<div class='muted'>{esc(sslinfo)}</div>")
        f.write("</div>")
        # Ports
        ports = data.get("ports", {})
        f.write("<div class='card'><h2>Port Scan</h2>")
        if "error" in ports:
            f.write(f"<div class='muted'>{esc(ports['error'])}</div>")
        else:
            f.write(f"<div class='muted'>Resolved IP: {esc(ports.get('ip',''))}</div>")
            if ports.get("open_ports"):
                f.write("<table><tr><th>Port</th><th>Service</th></tr>")
                for p, svc in ports["open_ports"]:
                    f.write(f"<tr><td>{esc(p)}</td><td>{esc(svc)}</td></tr>")
                f.write("</table>")
            else:
                f.write("<div class='muted'>No common open ports detected.</div>")
        f.write("</div>")
        f.write("</body></html>")
    return filename

# ========= SINGLE DOMAIN SCAN ==========
def scan_single(domain, do_brute=False, brute_wordlist=None, save_html=False):
    output_buffer.clear()
    log(BANNER, store=False)
    log(f"{C.YELLOW}ReconX scan started for: {domain}{C.RST}")
    data = {}
    try:
        subs = check_subdomains(domain)
        data["subdomains"] = subs
        if subs:
            log(f"{C.GREEN}--- Found Subdomains ({len(subs)}) ---{C.RST}")
            for h, ip in subs:
                log(f"{C.CYAN}{h}{C.RST} -> {C.MAG}{ip}{C.RST}")
        else:
            log(f"{C.RED}No common subdomains found.{C.RST}")
        if do_brute and brute_wordlist:
            log(f"\n{C.YELLOW}Starting brute-force subdomain scan using: {brute_wordlist}{C.RST}")
            brute_res = brute_subdomains(domain, brute_wordlist)
            data["brute"] = brute_res
            if brute_res:
                log(f"{C.GREEN}--- Brute-force Found ({len(brute_res)}) ---{C.RST}")
                for h, ips in brute_res:
                    log(f"{C.CYAN}{h}{C.RST} -> {C.MAG}{', '.join(ips)}{C.RST}")
            else:
                log(f"{C.RED}No brute-forced subdomains found.{C.RST}")
        else:
            data["brute"] = []
        log(f"\n{C.YELLOW}Retrieving SSL certificate...{C.RST}")
        ssl_info = get_ssl_certificate(domain)
        data["ssl"] = ssl_info
        if isinstance(ssl_info, dict) and "error" not in ssl_info:
            subj = ssl_info.get("subject", ())
            issuer = ssl_info.get("issuer", ())
            log(f"{C.GREEN}--- SSL certificate retrieved ---{C.RST}")
            log(f"{C.MAG}Subject:{C.RST} {subj}")
            log(f"{C.MAG}Issuer:{C.RST} {issuer}")
            log(f"{C.MAG}NotBefore:{C.RST} {ssl_info.get('notBefore')}")
            log(f"{C.MAG}NotAfter:{C.RST} {ssl_info.get('notAfter')}")
        else:
            log(f"{C.RED}SSL info error: {ssl_info.get('error') if isinstance(ssl_info, dict) else ssl_info}{C.RST}")
        log(f"\n{C.YELLOW}Scanning common ports...{C.RST}")
        ports = scan_ports(domain)
        data["ports"] = ports
        if "error" in ports:
            log(f"{C.RED}Port scan error: {ports['error']}{C.RST}")
        else:
            if ports.get("open_ports"):
                log(f"{C.GREEN}--- Open Ports ({len(ports['open_ports'])}) ---{C.RST}")
                for p, svc in ports["open_ports"]:
                    log(f"{C.CYAN}Port {p}{C.RST} ({C.MAG}{svc}{C.RST}) is open.")
            else:
                log(f"{C.RED}No common open ports detected.{C.RST}")
        log(f"\n{C.YELLOW}ReconX scan complete for: {domain}{C.RST}")
    except Exception as e:
        log(f"{C.RED}Unhandled error during scan: {e}{C.RST}")
        traceback.print_exc()
    # Write reports
    txt = write_text_report(domain, data)
    log(f"{C.YELLOW}Text report: {txt}{C.RST}")
    html_file = None
    if save_html:
        html_file = write_html_report(domain, data)
        log(f"{C.YELLOW}HTML report: {html_file}{C.RST}")
    return {"text": txt, "html": html_file, "data": data}

# ========= BATCH SCAN (multi-domain) ==========
def batch_scan(domains_file, do_brute=False, wordlist=None, save_html=False):
    if not os.path.isfile(domains_file):
        raise FileNotFoundError(f"Domains file not found: {domains_file}")
    results = []
    with open(domains_file, "r", errors="ignore") as fh:
        lines = [ln.strip() for ln in fh if ln.strip()]
    log(f"{C.YELLOW}Starting batch scan for {len(lines)} domains...{C.RST}")
    for d in lines:
        log(f"\n{C.BLUE}==> Scanning: {d}{C.RST}")
        res = scan_single(d, do_brute=do_brute, brute_wordlist=wordlist, save_html=save_html)
        results.append((d, res))
    log(f"\n{C.YELLOW}Batch scan complete.{C.RST}")
    return results

# ========= INTERACTIVE MENU ==========
def interactive():
    clear_cmd = 'cls' if os.name == 'nt' else 'clear'
    os.system(clear_cmd)
    print(BANNER)
    print(f"{C.MAG}Choose an option:{C.RST}")
    print("  1) Single domain quick scan")
    print("  2) Single domain + brute-force subdomains (wordlist)")
    print("  3) Multi-domain batch scan (file)")
    print("  4) Multi-domain batch + brute (file + wordlist)")
    print("  5) Exit")
    choice = input(f"\n{C.CYAN}Option: {C.RST}").strip()
    if choice == "1":
        d = input("Enter domain (example.com): ").strip()
        h = input("Save HTML report? (y/N): ").strip().lower() == 'y'
        scan_single(d, do_brute=False, brute_wordlist=None, save_html=h)
    elif choice == "2":
        d = input("Enter domain (example.com): ").strip()
        wl = input("Path to wordlist file: ").strip()
        h = input("Save HTML report? (y/N): ").strip().lower() == 'y'
        if not os.path.isfile(wl):
            log(f"{C.RED}Wordlist file not found: {wl}{C.RST}")
            return
        scan_single(d, do_brute=True, brute_wordlist=wl, save_html=h)
    elif choice == "3":
        df = input("Path to domains file (one domain per line): ").strip()
        h = input("Save HTML reports? (y/N): ").strip().lower() == 'y'
        if not os.path.isfile(df):
            log(f"{C.RED}Domains file not found: {df}{C.RST}")
            return
        batch_scan(df, do_brute=False, wordlist=None, save_html=h)
    elif choice == "4":
        df = input("Path to domains file: ").strip()
        wl = input("Path to wordlist file: ").strip()
        h = input("Save HTML reports? (y/N): ").strip().lower() == 'y'
        if not os.path.isfile(df):
            log(f"{C.RED}Domains file not found: {df}{C.RST}")
            return
        if not os.path.isfile(wl):
            log(f"{C.RED}Wordlist file not found: {wl}{C.RST}")
            return
        batch_scan(df, do_brute=True, wordlist=wl, save_html=h)
    elif choice == "5":
        log("Exiting.")
        sys.exit(0)
    else:
        log(f"{C.RED}Invalid choice.{C.RST}")

# ========= UTILITY ==========
def print_usage_and_exit():
    print("Usage: python3 reconx.py")
    print("Then follow the interactive prompts.")
    sys.exit(0)

# ========= MAIN ==========
if __name__ == "__main__":
    try:
        interactive()
    except KeyboardInterrupt:
        log(f"\n{C.RED}User aborted. Exiting.{C.RST}")
        sys.exit(1)
    except Exception as e:
        log(f"{C.RED}Fatal error: {e}{C.RST}")
        traceback.print_exc()
        sys.exit(1)
